<?php
header('Content-Type: application/json');
require 'vendor/autoload.php';
date_default_timezone_set('Asia/Dhaka');
include('database.php');

$uploadDir = __DIR__ . '/uploads/';
$imageDir = __DIR__ . '/images/';

// Create necessary directories if they don't exist
if (!file_exists($uploadDir)) mkdir($uploadDir, 0777, true);
if (!file_exists($imageDir)) mkdir($imageDir, 0777, true);

// Check if a file was uploaded
if (!isset($_FILES['pdf']) || $_FILES['pdf']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(['error' => 'No file uploaded or upload error occurred.']);
    exit;
}

// Move uploaded PDF to the server
$fileTmpPath = $_FILES['pdf']['tmp_name'];
$uploadedFilePath = $uploadDir . md5(uniqid()) . '.pdf';

if (!move_uploaded_file($fileTmpPath, $uploadedFilePath)) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to move uploaded file.']);
    exit;
}

// Function to extract images from PDF
function extractImagesFromPdf($pdfFilePath, $nid) {
    $parser = new \Smalot\PdfParser\Parser();
    $pdf = $parser->parseFile($pdfFilePath);
    $images = $pdf->getObjectsByType('XObject', 'Image');

    global $imageDir;
    $imageNames = [];
    $counter = 0;

    foreach ($images as $image) {
        $imageData = $image->getContent();
        $tempPath = $imageDir . 'temp_' . uniqid() . '.tmp';
        file_put_contents($tempPath, $imageData);
        $imageInfo = getimagesize($tempPath);

        if (!$imageInfo) {
            unlink($tempPath);
            continue;
        }

        $extension = image_type_to_extension($imageInfo[2], false) ?: 'png';
        $filename = ($counter === 0) ? "{$nid}-user.{$extension}" : "{$nid}-sign.{$extension}";
        rename($tempPath, $imageDir . $filename);

        $imageNames[$counter] = $filename;
        $counter++;
    }

    return $imageNames;
}

// function to download image
function downloadImage($url, $savePath) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30
    ]);
    $data = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);

    if ($err || !$data) return false;

    file_put_contents($savePath, $data);
    return true;
}

// Function to call the API if signature image is missing
function fetchSignatureFromAPI($pdfFilePath, $nid) {
    global $imageDir;

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => "https://pythoncode.unknownx1337.dev/imageextract/images",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => ['file' => new CURLFile($pdfFilePath)],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false
    ]);

    $response = curl_exec($ch);
    $curlErr = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if (!$response || $curlErr || $httpCode >= 400) return null;

    $json = json_decode($response, true);
    if (empty($json['images']['sign-image'])) return null;

    // Fetch the sign-image URL from the API response
    $signImageUrl = $json['images']['sign-image'];
    $imageExt = pathinfo(parse_url($signImageUrl, PHP_URL_PATH), PATHINFO_EXTENSION) ?: 'jpg';
    $imageName = "{$nid}-sign.{$imageExt}";
    $imagePath = $imageDir . $imageName;
    
    if (!downloadImage($signImageUrl, $imagePath)) return null;

    return file_exists($imagePath) ? $imageName : null;
}

// Function to extract text from PDF
function extractPdfData($filePath) {
    try {
        $parser = new Smalot\PdfParser\Parser();
        $pdf = $parser->parseFile($filePath);
        $text = $pdf->getText();

        $name = extractField($text, "Name(Bangla)", "Name(English)");
        $nid = extractField($text, "National ID", "Pin");

        $imageNames = extractImagesFromPdf($filePath, $nid);
        
        // If signature image is missing, try to fetch from API
        if (count($imageNames) < 2) {
            $signImg = fetchSignatureFromAPI($filePath, $nid);
            
            if ($signImg) {
                $imageNames[1] = $signImg;
            }
        } else {
            $signImg = $imageNames[1];
        }

        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
        $domain = $_SERVER['SERVER_NAME'];

        $dobRaw = extractField($text, "Date of Birth", "Birth Place");
        $dobRaw = trim(preg_replace('/[^\d\-]/', '', $dobRaw));
        $dateOfBirth = (strtotime($dobRaw) !== false) ? date('d M Y', strtotime($dobRaw)) : null;

        $preAddText = extractField($text, "Present", "Permanent");
        $perAddText = extractField($text, "Permanent", "Education");
        
        #echo($perAddText);

        $data = [
            "nameBn" => $name,
            "nameEn" => strtoupper(extractField($text, "Name(English)", "Date of Birth")),
            "nationalId" => $nid,
            "pin" => extractField($text, "Pin", "Status"),
            "dateOfBirth" => $dateOfBirth,
            "dateOfToday" => convertToBangla(date('d-m-Y', time())),
            "father" => extractField($text, "Father Name", "Mother Name") ?? extractField($text, "Father Name", "\n"),
            "mother" => extractField($text, "Mother Name", "Spouse Name") ?? extractField($text, "Mother\nName", "Spouse"),
            "spouse" => extractField($text, "Spouse Name", "\n") ?: extractField($text, "Spouse\nName", "Gender"),
            "gender" => extractField($text, "Gender", "Marital"),
            "religion" => extractField($text, "Religion", "\n") ?: extractField($text, "Religion", "Religion"),
            "birthPlace" => extractField($text, "Birth Place", "Birth Other"),
            "bloodGroup" => extractField($text, "Blood Group", "TIN"),
            "voterArea" => extractField($text, "Voter Area", "\n") ?: extractField($text, "Voter Area", "Voter"),
            "voterAreaCode" => extractField($text, "Voter Area Code", "\n"),
            "voterNo" => str_replace(["\t", "N", "\n", "Voter", "o", " "], "", extractField($text, "Lock Flag", "Form No")) ?: extractField($text, "Voter No", "\n") ?: extractField($text, "Voter No", "Form No"),
            "slNo" => extractField($text, "Sl No", "\n") ?: extractField($text, "Sl No", "Tag"),
            "occupation" => extractField($text, "Occupation", "\n") ?: extractField($text, "Occupation", "Disability"),
            "education" => extractField($text, "Education", "\n") ?: extractField($text, "Education", "Education"),
            "photo" => $protocol . $domain . "/SIGNTOSV/images/" . $imageNames[0],
            "sign" => isset($signImg) ? $protocol . $domain . "/SIGNTOSV/images/" . $signImg : 'N/A',
            "permanentAddress" => combineAddress($perAddText . " A"),
            "presentAddress" => combineAddress($preAddText . " A")
        ];
        
        return [
            "code" => 200,
            "success" => true,
            "message" => "Data fetched successfully",
            "data" => $data
        ];
    } catch (Exception $e) {
        return [
            "code" => 500,
            "success" => false,
            "message" => "Error processing PDF: " . $e->getMessage()
        ];
    }
}

// Function to extract text fields
function extractField($text, $start, $end) {
    $pattern = '/' . preg_quote($start, '/') . '(.*?)' . preg_quote($end, '/') . '/s';
    if (preg_match($pattern, $text, $matches)) {
        return trim($matches[1]);
    }
    return null;
}

// Function to convert numbers to Bangla
function convertToBangla($number) {
    $englishDigits = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    $banglaDigits = ['০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'];
    return str_replace($englishDigits, $banglaDigits, $number);
}

function combineAddress($text) {
    $home = extractField($text, "Home/Holding", "Post");
    $village = extractField($text, "Village/Road", "Home/Holding");

    // Convert text encoding to UTF-8 to fix corruption
    $text = mb_convert_encoding($text, 'UTF-8', 'auto');

    // Extract Mouza/Moholla
    $mouza = extractField($text, "Mouza/Moholla", "Ward For Union");
    if (empty(trim($mouza)) || strpos($mouza, "�") !== false) {
        $mouza = extractField($text, "Additional Mouza/Moholla", "Ward For Union");
        if (empty($mouza)) {
            $mouza = extractField($text, "Additional\nMouza/Moholla", "Ward");
        }
    }
    $mouza = trim(preg_replace('/�+/u', '', str_replace(["Mouza/Moholla", "Additional"], "", $mouza)));

    $wfup = extractField($text, "Porishod", "Village/Road");

    $unionOrWard = extractField($text, "Union/Ward", "\n");
    $postOffice = extractField($text, "Post Office", "Postal Code") ?? extractField($text, "Post\nOffice", "Postal\nCode");
    $postalCode = extractField($text, "Postal Code", "Region") ?? extractField($text, "Postal\nCode", "Region");
    
    $region = ($r = extractField($text, "Region", "A")) && strpos($r, "\n") !== false ? extractField($text, "Region", "\n") : $r;
    $division = extractField($text, "Division", "District");
    $district = extractField($text, "District", "RMO");
    
    $rmo = extractField($text, "RMO", "City");

    // Extract Municipality
    $text = preg_replace('/City\s*Corporation\s*Or\s*\n\s*Municipality/u', 'City Corporation Or Municipality', $text);
    $municipality = extractField($text, "City Corporation Or Municipality", "Upozila");
    $municipality = trim(preg_replace('/�+/u', '', $municipality));
    $municipality = preg_replace('/\bUpozila\b.*/u', '', $municipality);

    $upozila = trim(str_replace("\n", " ", extractField($text, "Upozila", "Union/Ward")));

    if (empty(trim($village)) || strpos($village, "�") !== false) {
        $village = extractField($text, "Additional Village/Road", "Home/Holding");
    }
    $village = trim(str_replace(["Village/Road", "Additional"], "", $village));

    $home = trim($home);
    $homeText = ($home === "No" || $home === "-" || $home === "," || $home === "") ? "" : "বাসা/হোল্ডিং:$home, ";
    $filteredhome = ($homeText === "বাসা/হোল্ডিং: ," || $homeText === "বাসা/হোল্ডিং:-,") ? "" : $homeText;
    
    $mouzaPart = !empty($mouza) ? "$mouza, " : "";
    $municipalityPart = !empty(trim($municipality)) ? "$municipality, " : "";

    $addL = $filteredhome
      . ", গ্রাম/রাস্তাঃ " . $village
      . ", মৌজা/মহল্লাঃ " . $mouza
      . ", ইউনিয়ন/ওয়ার্ডঃ " . $unionOrWard
      . ", পোস্ট অফিসঃ " . $postOffice
      . ", পোস্ট কোডঃ " . convertToBangla($postalCode)
      . ", উপজেলাঃ " . $upozila
      . ", জেলাঃ " . $district
      . ", অঞ্চলঃ " . $region
      . ", বিভাগঃ " . $division . "।";
    
    $addL = str_replace(["\t", "\n", "  ", 'Additional', 'No'], ['', ' ', " ", '', ''], $addL);

    $address = [
            "division" => $division,
            "district" => $district,
            "region" => $region,
            "rmo" => trim(str_replace("\n", " ", $rmo)),
            "upozila" => $upozila,
            "postOffice" => trim(str_replace("\n", " ", $postOffice)),
            "postCode" => $postalCode,
            "unionOrWard" => $unionOrWard,
            "wardForUnionPorishod" => $wfup,
            "mouzaOrMoholla" => $mouzaPart,
            "villageOrRoad" => trim(str_replace("\n", " ", $village)),
            "houseOrHoldingNo" => trim(str_replace(['No', "\n"], ['', ' '], $home)),
            "addressLine" => trim($addL)
        ];
    return $address;
}

$key = $_POST['key'] ?? '';

if (empty($key)) {
    http_response_code(400);
    echo json_encode([
        "success" => false,
        "code" => 400,
        "message" => "Missing required parameter key"
    ], JSON_PRETTY_PRINT);
    exit;
}

$sql = "SELECT * FROM users WHERE apiKey = ?";
if ($stmt = $conn->prepare($sql)) {
    $stmt->bind_param("s", $key);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
        $expDate = $user['expiryDate'];
        $status = strtolower($user['status']);
        $balance = intval($user['balance']);
        $currentDateTime = date('Y-m-d H:i:s');

        if (strtotime($expDate) < strtotime($currentDateTime)) {
            sendResponse(403, 'API key expired', 'Your API key has expired. Please contact admin to renew.');
        }

        if ($status !== 'on') {
            sendResponse(403, 'This key is deactivated.', 'Contact Admin To Activate Key');
        }

        if ($balance <= 0 && $balance !== 99999999) {
            sendResponse(403, 'Insufficient balance.', 'Contact Admin To Buy Balance');
        }

        $result = extractPdfData($uploadedFilePath);

        if ($result['success'] && $balance > 0) {
            if ($balance !== 99999999) {
                $updateSql = "UPDATE users SET balance = balance - 1, used = used + 1 WHERE apiKey = ?";
            } else {
                $updateSql = "UPDATE users SET balance = balance, used = used + 1 WHERE apiKey = ?";
            }
        }

        if ($updateStmt = $conn->prepare($updateSql)) {
            $updateStmt->bind_param("s", $key);
            $updateStmt->execute();
            $updateStmt->close();
        }

    echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    } else {
        sendResponse(401, 'Invalid API key.', 'Unauthorized access');
    }

    $stmt->close();
} else {
    sendResponse(500, 'Database error!', 'An unexpected error occurred. Please report this to the admin.');
}

$conn->close();

function sendResponse($status, $error, $details) {
    http_response_code($status);
    echo json_encode([
        'success' => false,
        'code' => $status,
        'error' => $error,
        'details' => $details
    ], JSON_PRETTY_PRINT);
    exit;
}

?>